import React, { Component } from 'react';
import electron from 'electron';
import path from 'path';
import { History } from 'history';
import { withStyles } from '@material-ui/core/styles';
import { Logger } from 'library/Logger';
import badgeIcon from 'resources/images/updated.svg';
import mainIcon from 'resources/images/cardo_unit.svg';
import routes from 'constants/routes';
import storage from 'azure-storage';
import moment from 'moment';
import {
    AZURE_BLOB_SERVICE_CONNECTION_STRING,
    AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME,
    AZURE_BLOB_SERVICE_LOGS_URL
} from 'constants/urls';
import styles from './styles';

type Props = {
    device: object,
    language: string,
    strings: object,
    history: History,
    classes: {
        [key: string]: string
    }
};

class UpdateDeviceCompleteScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            device: props.device,
            language: props.language,
            strings: props.strings
        };
    }

    componentDidMount() {
        Logger.info('UpdateDeviceCompleteScreen: Mounted');

        this.sendLogs();

        setTimeout(() => {
            this.moveToMainScreen();
        }, 5000);
    }

    componentDidUpdate(prevProps, prevState) {
        const { language, strings, device } = this.props;
        if (prevState.language !== language) {
            this.setState({ language, strings });
        }
        if (prevState.device.connected && !device.connected) {
            // Device disconnected
            this.moveToMainScreen();
        }
    }

    moveToMainScreen() {
        const { history } = this.props;
        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            // Move to main screen, but do not perform auto update checks again
            history.replace({
                pathname: routes.WELCOME.path,
                state: { skipAutoUpdate: true }
            });
        }, 100);
    }

    async sendLogs() {
        // Send over the log files
        const logDir = electron.remote.app.getPath('userData');
        const appLogPath = path.join(logDir, 'updater.log');
        const libraryLogPath = path.join(logDir, 'cfu.log');

        const blobService = storage.createBlobService(
            AZURE_BLOB_SERVICE_CONNECTION_STRING
        );
        const blockBlobContainerName = AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME;

        const currentTime = moment().format('YYYY-MM-DD_HHmmss');
        const uuid =
            Math.random()
                .toString(36)
                .substring(2, 15) +
            Math.random()
                .toString(36)
                .substring(2, 15);

        const appLogFileBlobName = `${currentTime}-${uuid}-app.log`;
        const libraryLogFileBlobName = `${currentTime}-${uuid}-cfu.log`;

        console.log('Uploading app log file', appLogFileBlobName);

        blobService.createBlockBlobFromLocalFile(
            blockBlobContainerName,
            appLogFileBlobName,
            appLogPath,
            error => {
                console.log('Finished uploading app log file', error);
            }
        );

        console.log('Uploading library log file', libraryLogFileBlobName);

        blobService.createBlockBlobFromLocalFile(
            blockBlobContainerName,
            libraryLogFileBlobName,
            libraryLogPath,
            error => {
                console.log('Finished uploading library log file', error);
            }
        );

        const baseUrl = `${AZURE_BLOB_SERVICE_LOGS_URL}/${AZURE_BLOB_SERVICE_LOGS_CONTAINER_NAME}`;

        console.log('Base URL', baseUrl);
    }

    render() {
        const { strings } = this.state;
        const { classes } = this.props;
        return (
            <div className={classes.container}>
                <div className={classes.innerContainer}>
                    <div className={classes.imageContainer}>
                        <img
                            className={classes.mainImage}
                            src={mainIcon}
                            alt="Main Icon"
                        />
                        <img
                            className={classes.badgeImage}
                            src={badgeIcon}
                            alt="Badge Icon"
                        />
                    </div>
                    <div className={classes.title}>
                        {strings.update_device_done_title}
                    </div>
                    <div className={classes.subtitle}>
                        {strings.update_device_done_subtitle}
                    </div>
                </div>
            </div>
        );
    }
}

export default withStyles(styles)(UpdateDeviceCompleteScreen);
